//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	Household and enumerated-person level aggregate net worth in HILDA
// Objective: 	Calculating aggregate net worth in HILDA, both at the household
//				level and the enumerated-person level using the created
//				individual wealth variables.
//
// Created: 	31/03/2021
// Modified: 	15/11/2021
//==============================================================================

**********************************
*** HILDA Household Level Data ***
**********************************

clear all	// clearing the workspace

set maxvar 12000 // setting the maximum number of variables to 12000 (default is 5000) to allow STATA to read in all observations

cd "XXXX"	// set content directory here

foreach x in b f j n r {
    use "Household_`x'190u.dta", clear	// HILDA wealth modules
	g assets = `x'hwassei 			// extracting the assets variable for each wave
	g debt = `x'hwdebti 			// extracting the debt variable for each wave
	g hh_net_worth = assets - debt 	// generating a net worth variable for each wave
	g bank_acc = `x'hwtbani 		// extracting the bank accounts variable for each wave
	g superannuation = `x'hwsupei 	// extracting the superannuation variable for each wave
	g cash_inv = `x'hwcaini 		// extracting the cash investments variable for each wave
	g equity_inv = `x'hweqini 		// extracting the equity investments variable for each wave
	g trust_funds = `x'hwtrusi 		// extracting the trust funds variable for each wave
	g life_insur = `x'hwinsui 		// extracting the life insurance variable for each wave
	g property_assets = `x'hwtpvi 	// extracting the property assets variable for each wave
	g bus_assets = `x'hwbusvi 		// extracting the business assets variable for each wave
	g collectibles = `x'hwcolli 	// extracting the collectibles variable for each wave
	g vehicles = `x'hwvechi 		// extracting the vehicles variable for each wave
	g credit_debt = `x'hwccdti 		// extracting the credit card debt variable for each wave
	g bus_debt = `x'hwbusdi 		// extracting the business debt variable for each wave
	g property_debt = `x'hwtpdi 	// extracting the property debt variable for each wave
	g wave = "`x'" 					// extracting the wave variable for each wave
	g hwt = `x'hhwth 				// extracting population level household weights for each wave
	g hwts = `x'hhwths 				// extracting sample level household weights for each wave
	g hhid = `x'hhrhid 				// extracting the household id variable for each wave
	keep assets debt hh_net_worth bank_acc superannuation cash_inv equity_inv trust_funds life_insur property_assets bus_assets collectibles vehicles credit_debt bus_debt property_debt wave hwt hwts hhid // keeping the relevant variables
	save "wealth_u_temp`x'.dta", replace
}

*****************************
*** Appending The Dataset ***
*****************************

use "wealth_u_tempb.dta", replace
append using "wealth_u_tempf.dta"
append using "wealth_u_tempj.dta"
append using "wealth_u_tempn.dta"
append using "wealth_u_tempr.dta"

**********************************
*** Erasing Temporary Datasets ***
**********************************

erase "wealth_u_tempb.dta"
erase "wealth_u_tempf.dta"
erase "wealth_u_tempj.dta"
erase "wealth_u_tempn.dta"
erase "wealth_u_tempr.dta"

*********************************
*** Generating Year Variables ***
*********************************

g year = 2002 if wave == "b"
replace year = 2006 if wave == "f"
replace year = 2010 if wave == "j"
replace year = 2014 if wave == "n"
replace year = 2018 if wave == "r"

destring hhid, replace // destringing the household id variable

save "HILDA_u_wealth.dta", replace	// saving the dataset

**********************************
*** Weighting Wealth Variables ***
**********************************

local HILDA_hh_wealth_vars assets debt hh_net_worth bank_acc superannuation cash_inv equity_inv trust_funds life_insur property_assets bus_assets collectibles vehicles credit_debt bus_debt property_debt	// creating a local macro containing household wealth variables

foreach w in `HILDA_hh_wealth_vars' {
	generate wt_`w' = `w' * hwt // weighting wealth variables by the population level household weights
}

table year, c(sum wt_hh_net_worth) // creating a table with the weighted aggregate net worth by year

collapse(sum) wt_assets wt_debt wt_hh_net_worth wt_bank_acc wt_superannuation wt_cash_inv wt_equity_inv wt_trust_funds wt_life_insur wt_property_assets wt_bus_assets wt_collectibles wt_vehicles wt_credit_debt wt_bus_debt wt_property_debt, by(year)	// collapsing the dataset to leave the sum each year

keep year wt_hh_net_worth	// keeping relevant variables

save Agg_household_wealth_HILDA.dta, replace	// saving the dataset

******************************************
*** HILDA Enumerated Person Level Data ***
******************************************

clear all	// clearing the workspace

cd "XXXX"	// set content directory here

use HILDA_restricted_combined_rescaled.dta, replace	// opening the dataset

keep if (wave == 2 | wave == 6 | wave == 10 | wave == 14 | wave == 18)	// keeping wealth module years

g ep_net_worth = pwassei - pwdebti	// generating an enumerated person net worth variable
g wt_ep_net_worth = ep_net_worth * hhwte	// applying the population-level enumerated person weights to the net worth variable

collapse(sum) wt_ep_net_worth, by(year)	// collapsing the dataset by year

save Agg_enumerated_person_wealth_HILDA.dta, replace	// saving the dataset

merge 1:1 year using Agg_household_wealth_HILDA, nogen	// merging in the aggregate household wealth variable

gen month = 6

save Agg_wealth_HILDA, replace	// saving the dataset